/* $Id: srcparb3.c,v 1.1 1998/09/24 01:10:19 donwm Exp $ */
/* Copyright (C) 1996 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Don Mathiesen */

/*
Example of changing the burst arb signal, during a measurment,
before manual arm.

The arb signal alternates between a sine wave, and a triangle wave.
Note that buffer A and buffer B have different frequencies.

This is derived fron srcarb2.c .
*/

#include <stdlib.h>		/* For exit */
#include <stdio.h>		/* For printf */
#include <unistd.h>		/* For sleep */
#include <sicl.h>		/* For SICL functions */
#include <math.h>
#include "e1432.h"

/* #define DPRINT */
/* #define DPRINT2 */

#define	BLOCKSIZE	8192

#define	SIGLENGTH	30000
#define	SIGPERIOD	7500/4
#define	XFRLENGTH	4096
#define	XFRLENGTHMAX	4096
#define SRCBUFRSIZE	15000
#define SRCBUFRSIZEMAX	40960/2
/* for single channel per sca SRCBUFRSIZEMAX is 40960 */
/*
   previous values with XFRLENGTH a sub multiple of SIGLENGTH, which is no
   longer a constraint
*/
/*
#define	SIGLENGTH	8192
#define	SIGPERIOD	8192/4
#define	XFRLENGTH	2048
#define	XFRLENGTHMAX	4096
#define	SRCBUFRSIZE	4096
#define	SRCBUFRSIZEMAX	4096
*/

#define MANUAL_ARM	1	/* 0 = autoarm/trig; 1 = manual arm/trig */

#define	SPAN		20000.0

#define	IN_MAX		32
#define	SRC_MAX		5

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif

/* Make these global  */
static E1432ID hw;
static SHORTSIZ16 group;
static SHORTSIZ16 chan_list[IN_MAX + SRC_MAX];
static int chan_count;

static SHORTSIZ16 in_list[IN_MAX];
static SHORTSIZ16 in_group;
static int in_count;

static SHORTSIZ16 src_list[SRC_MAX];
static SHORTSIZ16 src_group;
static int src_count;


/******************* arb source routines **************************/

#define SIG_LENGTH_MAX 100000

static LONGSIZ32 src_sigbuff1_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff2_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff3_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff4_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff5_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff1_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff2_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff3_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff4_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff5_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_siglength;

static LONGSIZ32 src_sigbuff1_a_tri[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff2_a_tri[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff3_a_tri[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff4_a_tri[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff5_a_tri[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff1_b_tri[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff2_b_tri[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff3_b_tri[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff4_b_tri[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff5_b_tri[SIG_LENGTH_MAX];

static int src_sigperiod;
static SHORTSIZ16 arb_xfrmode, arb_xfrmodewait;

static SHORTSIZ16 arb_mode;
static LONGSIZ32 arb_srcbufsize;
static LONGSIZ32 arb_xfrsize;

void
src_sig( LONGSIZ32 siglength, int sigperiod, long *sigbuff)
{

int i;

	  /* Synth a sine */
          for(i=0;i<siglength;i++)
	  {
             sigbuff[i] = (long)((float) 0x7fffffffL *
                          sin(2.0*M_PI*(float)i/((float)sigperiod)));
          }
}

void
src_sig_tri( LONGSIZ32 siglength, int sigperiod, long *sigbuff)
{

int i;
float p;

	  /* Synth a triangle */
          p =0;

          for(i=0;i<siglength;i++)
	  {
             if (p > (float)sigperiod) p = p - (float)sigperiod;

             if (p < ((float)sigperiod)*.25)
	     {
		 sigbuff[i] = (long)((float) 0x7fffffffL *
                          p/(((float)sigperiod)*.25));
		 goto next_p;
	     }
             if (p < ((float)sigperiod)*.75)
	     {
		 sigbuff[i] = (long)((float) 0x7fffffffL *
                          (2. - (p/(((float)sigperiod)*.25))));
		 goto next_p;
	     }
             if (p < (float)sigperiod)
	     {
		 sigbuff[i] = (long)((float) 0x7fffffffL *
                          (-4. + (p/(((float)sigperiod)*.25))));
	     }
	     
next_p:
	     p=p+1;
	     
	 }

}


void
xfr_src_sig(E1432ID lhw, SHORTSIZ16 chanID, long *sigbuff, int xfrlength, SHORTSIZ16 mode)
{
    LONGSIZ32 i, maxtries;
    short error;
    SHORTSIZ16 status;
    LONGSIZ32 xfrbuf_wds,srcbuf_state_AB,srcbuf_state_A,srcbuf_state_B;
   
	/* write xfrlength words to the substrate */

	i = 0;
	maxtries = 10;

	for (i = 0; i < maxtries; i++)
	{
#ifdef DPRINT2
	    e1432_get_src_arbstates(lhw, chanID, &xfrbuf_wds, &srcbuf_state_AB, &srcbuf_state_A, &srcbuf_state_B);
	    (void) printf("xfrbuf_wds %d, srcbuf_state_AB 0x%x, srcbuf_state_A 0x%x, srcbuf_state_Bxf 0x%x\n",xfrbuf_wds,srcbuf_state_AB,srcbuf_state_A, srcbuf_state_B);
#endif	    
	    status = e1432_check_src_arbrdy(lhw, chanID, mode);
#ifdef DPRINT2
	    (void) printf("xfr_src_sig: status 0x%x\n",status);
#endif
	    
	    if (status == 1)
	    {
		error = e1432_write_srcbuffer_data(lhw, chanID,
						  sigbuff,
						  (SHORTSIZ16)  xfrlength,
						  mode);
		if(error)
		{
		    (void) printf("Error %d in e1432_write_srcbuffer_data\n",
				  error);
		    e1432_get_src_arbstates(lhw, chanID, &xfrbuf_wds, &srcbuf_state_AB, &srcbuf_state_A, &srcbuf_state_B);
		    (void) printf("xfrbuf_wds %d, srcbuf_state_AB 0x%x, srcbuf_state_A 0x%x, srcbuf_state_Bxf 0x%x\n",xfrbuf_wds,srcbuf_state_AB,srcbuf_state_A, srcbuf_state_B);
	    
		    status = e1432_check_src_arbrdy(lhw, chanID, mode);
		    (void) printf("xfr_src_sig: status 0x%x\n",status);
		}
#ifdef DPRINT2
		e1432_get_src_arbstates(lhw, chanID, &xfrbuf_wds, &srcbuf_state_AB, &srcbuf_state_A, &srcbuf_state_B);
		(void) printf("xfrbuf_wds %d, srcbuf_state_AB 0x%x, srcbuf_state_A 0x%x, srcbuf_state_Bxf 0x%x\n",xfrbuf_wds,srcbuf_state_AB,srcbuf_state_A, srcbuf_state_B);
	    
		status = e1432_check_src_arbrdy(lhw, chanID, mode);
		(void) printf("xfr_src_sig: status 0x%x\n",status);
#endif
		return;
	    }
	}
	if (i == maxtries)
	{
	    (void) printf("xfr_src_sig: timed out \n");
	    e1432_get_src_arbstates(lhw, chanID, &xfrbuf_wds, &srcbuf_state_AB, &srcbuf_state_A, &srcbuf_state_B);
	    (void) printf("xfrbuf_wds %d, srcbuf_state_AB 0x%x, srcbuf_state_A 0x%x, srcbuf_state_Bxf 0x%x\n",xfrbuf_wds,srcbuf_state_AB,srcbuf_state_A, srcbuf_state_B);
	    
	    status = e1432_check_src_arbrdy(lhw, chanID, mode);
	    (void) printf("xfr_src_sig: status 0x%x\n",status);
	    return;
	}
}

/***************** end arb source routines ************************/

static int
read_data(void)
{
    FLOATSIZ64 buffer[BLOCKSIZE];
    LONGSIZ32 count;
    int     i;

    (void) printf("read_data\n");
    /* Read some data */
    for (i = 0; i < in_count; i++)
    {
	CHECK(e1432_read_float64_data(hw, in_list[i],
				      E1432_TIME_DATA, buffer,
				      BLOCKSIZE, NULL, &count));
	 (void) printf("channel %d    data points read =  %d\n", i , count);
	if (count != BLOCKSIZE)
	{
	    DEBUG((void) printf("Actual count was %d\n", count));
	    return -1;
	}
    }
    return 0;
}
 
int
main(void)
{
    int     i, error;
    struct e1432_hwconfig hwconfig;
    SHORTSIZ16 laddr = 8;
    int wait, nowait, manual_arm, signal;
    int doitforever;
    SHORTSIZ16 src_ch_state[5],meas_state;
    int src_count_total, k;
    LONGSIZ32 *src_sigbuffptrs_a[5];   
    LONGSIZ32 *src_sigbuffptrs_b[5];   
    LONGSIZ32 *src_sigbuffptrs_a_tri[5];   
    LONGSIZ32 *src_sigbuffptrs_b_tri[5];   
    LONGSIZ32 arb_xfrsize2,arb_srcbufleft;
    LONGSIZ32 arb_srcbufsize2;

    wait = 1;
    nowait = 0;
    manual_arm = MANUAL_ARM;
    src_sigbuffptrs_a[0]= &src_sigbuff1_a[0];   
    src_sigbuffptrs_a[1]= &src_sigbuff2_a[0];   
    src_sigbuffptrs_a[2]= &src_sigbuff3_a[0];   
    src_sigbuffptrs_a[3]= &src_sigbuff4_a[0];   
    src_sigbuffptrs_a[4]= &src_sigbuff5_a[0];   
    src_sigbuffptrs_b[0]= &src_sigbuff1_b[0];   
    src_sigbuffptrs_b[1]= &src_sigbuff2_b[0];   
    src_sigbuffptrs_b[2]= &src_sigbuff3_b[0];   
    src_sigbuffptrs_b[3]= &src_sigbuff4_b[0];   
    src_sigbuffptrs_b[4]= &src_sigbuff5_b[0];   
    src_sigbuffptrs_a_tri[0]= &src_sigbuff1_a_tri[0];   
    src_sigbuffptrs_a_tri[1]= &src_sigbuff2_a_tri[0];   
    src_sigbuffptrs_a_tri[2]= &src_sigbuff3_a_tri[0];   
    src_sigbuffptrs_a_tri[3]= &src_sigbuff4_a_tri[0];   
    src_sigbuffptrs_a_tri[4]= &src_sigbuff5_a_tri[0];   
    src_sigbuffptrs_b_tri[0]= &src_sigbuff1_b_tri[0];   
    src_sigbuffptrs_b_tri[1]= &src_sigbuff2_b_tri[0];   
    src_sigbuffptrs_b_tri[2]= &src_sigbuff3_b_tri[0];   
    src_sigbuffptrs_b_tri[3]= &src_sigbuff4_b_tri[0];   
    src_sigbuffptrs_b_tri[4]= &src_sigbuff5_b_tri[0];   
      
    /* Initialize the library */
    CHECK(e1432_init_io_driver());

    /* Change this 0 to 1 to see call tracing */
    e1432_trace_level(0);

    /* Use e1432_get_hwconfig to see if the module already has
       firmware.  If this errors, assume we need to install firmware,
       so use e1432_install to do it. */
    error = e1432_get_hwconfig(1, &laddr, &hwconfig);
    if (error != 0)
    {
	(void) printf("downloading /opt/e1432/lib/sema.bin\n");
        error = e1432_install(1, &laddr, 0,
                              "/opt/e1432/lib/sema.bin");
        if (error)
	{
            (void) printf("e1432_install failed, error: %d\n", error);
	    return -1;
	}
	
	CHECK(e1432_get_hwconfig(1, &laddr, &hwconfig));
    }

    /* Turn on debugging prints, a good idea while developing.  But do
       it after the above e1432_get_hwconfig, because we know that
       might error if the firmware is not yet installed in the
       module. */
    (void) e1432_print_errors(1);

    e1432_debug_level(0);

    CHECK(e1432_assign_channel_numbers(1, &laddr, &hw));
    (void) printf("hwconfig.input_chans %d\n",hwconfig.input_chans);
    (void) printf("hwconfig.source_chans %d\n",hwconfig.source_chans);
    /* Create input channel group */
    if ( hwconfig.input_chans > 0)
    {
	in_count = hwconfig.input_chans;
	if (in_count > IN_MAX)
	    in_count = IN_MAX;
	for (i = 0; i < in_count; i++)
	    in_list[i] = E1432_INPUT_CHAN(i+1);
	in_group = e1432_create_channel_group(hw, in_count, in_list);
	
	if (in_group >= 0)
	{
	    DEBUG((void) printf("e1432_create_channel_group returned %d\n",
				in_group));
	    return -1;
	}
    }
    else
    {
	in_count =0;
    }

    /* Create source group */
    if ( hwconfig.source_chans > 0)
    {
	src_count = hwconfig.source_chans;
	if (src_count > SRC_MAX)
	    src_count = SRC_MAX;
#ifdef DPRINT2
	(void) printf("src_count %d\n",src_count);
#endif
	k = 0;
	for (i = 0; i < src_count; i++)
	{
	    /* for all source channels : */
	    /* src_list[i] = E1432_SOURCE_CHAN(i+1); */
	    /* k = i+1; */

	    /* for 1 channel per sca : */
#ifdef DPRINT2
	    (void) printf("i %d\n",i);
	    (void) printf("k %d\n",k);
#endif
	    if (((i+1) == 1) || ((i+1) == 2) || ((i+1) == 3) || ((i+1) == 4) || ((i+1) == 5))

	    {
		(void) printf("adding channel %d to src_list\n",i+1);
		src_list[k] = E1432_SOURCE_CHAN(i+1);
		k++;
	    }
	}
	    src_count_total = src_count;
	    src_count = k;
	
	src_group = e1432_create_channel_group(hw, src_count, src_list);
	if (src_group >= 0)
	{
	    DEBUG((void) printf("e1432_create_channel_group returned %d\n",
			    src_group));
	    return -1;
	}
    }
    else
    {
	src_count =0;
    }

    /* Create global channel group */
    chan_count = 0;
    if (in_count > 0);
    {
	for (i = 0; i < in_count; i++)
	{
	    chan_list[chan_count] = in_list[i];
	    chan_count = chan_count+1;
	}
    }
    
    if (src_count > 0);
    {
	for (i = 0; i < src_count; i++)
	{
	    chan_list[chan_count] = src_list[i];
	    chan_count = chan_count+1;
	}
    }
    
    if (chan_count > 0)
    { 
	group = e1432_create_channel_group(hw, chan_count, chan_list);
	if (group >= 0)
	{
	    DEBUG((void) printf("e1432_create_channel_group returned %d\n",
				group));
	    exit(1);
	}
    }
    (void) printf("in_count %d\n",in_count);
    (void) printf("src_count_total %d\n",src_count_total);
    (void) printf("src_count %d\n",src_count);
    (void) printf("chan_count %d\n",chan_count);

/* initialize all channels */
    CHECK(e1432_set_clock_freq(hw, group,51200));
    CHECK(e1432_set_span(hw, group, SPAN));
    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));

    (void) printf("manual_arm (1=manual, 0 = auto) %d\n",manual_arm);
    if (manual_arm)
    {
	CHECK(e1432_set_auto_trigger(hw, group, E1432_MANUAL_TRIGGER));
	CHECK(e1432_set_auto_arm(hw, group, E1432_MANUAL_ARM));
    }
    else
    {
	CHECK(e1432_set_auto_trigger(hw, group, E1432_AUTO_TRIGGER));
	CHECK(e1432_set_auto_arm(hw, group, E1432_AUTO_ARM));
    }

/* initialize inputs */
     if (in_count > 0)
    {
	CHECK(e1432_set_active(hw, in_group, E1432_CHANNEL_ON));
	CHECK(e1432_set_data_mode(hw, in_group, E1432_BLOCK_MODE));
    }
    
/* initialize source */
    if (src_count > 0)
    {

	arb_mode = E1432_SRCBUFFER_PERIODIC_AB;
/*
	arb_mode = E1432_SRCBUFFER_PERIODIC_A;
*/

	if ((arb_mode == E1432_SRCBUFFER_PERIODIC_AB)||(arb_mode == E1432_SRCBUFFER_CONTINUOUS))
	{
	    arb_xfrmode = E1432_SRC_DATA_MODE_AB;
	    arb_xfrmodewait = E1432_SRC_DATA_MODE_WAITAB;
	}
	else
	{
	    arb_xfrmode = E1432_SRC_DATA_MODE_A;
	    arb_xfrmodewait = E1432_SRC_DATA_MODE_WAITA;
	}
	arb_srcbufsize = SRCBUFRSIZE;
	src_siglength = SIGLENGTH;
	src_sigperiod = SIGPERIOD;
	arb_xfrsize = XFRLENGTH;
#ifdef DPRINT2
	(void) printf("arb_xfrsize %d, arb_srcbufsize %d, src_siglength %d\n",
		      arb_xfrsize, arb_srcbufsize, src_siglength);
	(void) printf("arb_xfrmode 0x%x, arb_xfrmodewait %d\n",
		      arb_xfrmode, arb_xfrmodewait);
#endif
	CHECK(e1432_set_active(hw, src_group, E1432_CHANNEL_ON));

	if (manual_arm)
	{
	    CHECK(e1432_set_source_mode(hw, src_group, E1432_SOURCE_MODE_BARB));
	    CHECK(e1432_set_duty_cycle(hw, src_group, 1.0));
	    if (arb_xfrmode == E1432_SRC_DATA_MODE_AB)
	    {
		CHECK(e1432_set_source_blocksize(hw, src_group, 2*SRCBUFRSIZE));
	    }
	    else
	    {
		CHECK(e1432_set_source_blocksize(hw, src_group, SRCBUFRSIZE));
	    }
	}
	else
	{
	    CHECK(e1432_set_source_mode(hw, src_group, E1432_SOURCE_MODE_ARB));
	    CHECK(e1432_set_source_blocksize(hw, src_group, SRCBUFRSIZE));
	}
	CHECK(e1432_set_ramp_rate(hw, src_group, 0));
	CHECK(e1432_set_range(hw, src_group, 1.0));
/*	CHECK(e1432_set_span(hw, src_group, 625)); */
	CHECK(e1432_set_amp_scale(hw, src_group, 1.0));

	CHECK(e1432_auto_zero(hw, src_group));

	/** create signals ***/
	for (i=0; i<src_count;i++)
	{	
	    src_sig(src_siglength, 2*src_sigperiod/(2*(i+1)), src_sigbuffptrs_a[i]);
	    src_sig(src_siglength, 2*src_sigperiod, src_sigbuffptrs_b[i]);
	    src_sig_tri(src_siglength, 2*src_sigperiod/(2*(i+1)), src_sigbuffptrs_a_tri[i]);
	    src_sig_tri(src_siglength, 2*src_sigperiod, src_sigbuffptrs_b_tri[i]);
	}
	
	/* initialize source buffers  */
	for (i=0; i<src_count;i++)
	{	
	    CHECK(e1432_set_srcbuffer_mode(hw, src_list[i], arb_mode));
	    CHECK(e1432_set_srcbuffer_size(hw, src_list[i], arb_srcbufsize));
	    CHECK(e1432_set_srcbuffer_init(hw, src_list[i],
					   E1432_SRCBUFFER_INIT_EMPTY));
	    CHECK(e1432_get_srcbuffer_size(hw, src_list[i],
					   &arb_srcbufsize2));
	    if (arb_srcbufsize2 != arb_srcbufsize)
	    {
		printf("Source buffer size changed from %d to %d \n",
		       arb_srcbufsize,arb_srcbufsize2);
		
		arb_srcbufsize = arb_srcbufsize2;
	    }
	}	
	/* down load arb source buffer A signal */
	/* (void) printf("preload buffer A\n"); */
	arb_xfrsize2 = arb_xfrsize;
	for (k=0; k<arb_srcbufsize;k = k + arb_xfrsize)
	{	
	    arb_srcbufleft = arb_srcbufsize - k;
	    if (arb_srcbufleft < arb_xfrsize) arb_xfrsize2 = arb_srcbufleft;
	    for (i=0; i<src_count;i++)
	    {	
		(void) printf("preload buffer A, chan %x, %d words\n",src_list[i],arb_xfrsize2);
		xfr_src_sig(hw, src_list[i], src_sigbuffptrs_a[i]+k,arb_xfrsize2,arb_xfrmodewait);
	    }
	}
	if (arb_xfrmode == E1432_SRC_DATA_MODE_AB)
	{
	    /* (void) printf("preload buffer B\n"); */
	    /* down load arb source buffer B signal */
	    arb_xfrsize2 = arb_xfrsize;
	    for (k=0; k<arb_srcbufsize;k = k + arb_xfrsize)
	    {	
		arb_srcbufleft = arb_srcbufsize - k;
		if (arb_srcbufleft < arb_xfrsize) arb_xfrsize2 = arb_srcbufleft;
		for (i=0; i<src_count;i++)
		{	
		    (void) printf("preload buffer B, chan %x, %d words\n",src_list[i],arb_xfrsize2);
		    
		    xfr_src_sig(hw, src_list[i],
				    src_sigbuffptrs_b[i]+k,arb_xfrsize2,arb_xfrmodewait);
		    
		}
	    }
	}
	(void) printf("done pre-loading buffer(s)\n");
    }
    else
    {
	(void) printf("Error : No source channels found\n");
	/* exit(1); */
    }
    
/* Start measurement */
    (void) printf("init_measure\n");
    CHECK(e1432_init_measure(hw, group));

    doitforever = 1;
    signal = -1;
    
    while(doitforever)
    {
	if (manual_arm)
	{
	    if (signal >= 0)
	    {
		/* download alternate signal */
		if (signal == 0)
		{
		    signal = 1;
		    /* download triangle */
		    src_sigbuffptrs_a[0]= &src_sigbuff1_a_tri[0];   
		    src_sigbuffptrs_a[1]= &src_sigbuff2_a_tri[0];   
		    src_sigbuffptrs_a[2]= &src_sigbuff3_a_tri[0];   
		    src_sigbuffptrs_a[3]= &src_sigbuff4_a_tri[0];   
		    src_sigbuffptrs_a[4]= &src_sigbuff5_a_tri[0];   
		    src_sigbuffptrs_b[0]= &src_sigbuff1_b_tri[0];   
		    src_sigbuffptrs_b[1]= &src_sigbuff2_b_tri[0];   
		    src_sigbuffptrs_b[2]= &src_sigbuff3_b_tri[0];   
		    src_sigbuffptrs_b[3]= &src_sigbuff4_b_tri[0];   
		    src_sigbuffptrs_b[4]= &src_sigbuff5_b_tri[0];   
		}
		else
		{
		    signal = 0;
		    /* download sine */
		    src_sigbuffptrs_a[0]= &src_sigbuff1_a[0];   
		    src_sigbuffptrs_a[1]= &src_sigbuff2_a[0];   
		    src_sigbuffptrs_a[2]= &src_sigbuff3_a[0];   
		    src_sigbuffptrs_a[3]= &src_sigbuff4_a[0];   
		    src_sigbuffptrs_a[4]= &src_sigbuff5_a[0];   
		    src_sigbuffptrs_b[0]= &src_sigbuff1_b[0];   
		    src_sigbuffptrs_b[1]= &src_sigbuff2_b[0];   
		    src_sigbuffptrs_b[2]= &src_sigbuff3_b[0];   
		    src_sigbuffptrs_b[3]= &src_sigbuff4_b[0];   
		    src_sigbuffptrs_b[4]= &src_sigbuff5_b[0];   
		}
		sleep (1);
		
		/* initialize source buffers  */
		for (i=0; i<src_count;i++)
		{	
		    CHECK(e1432_set_srcbuffer_init(hw, src_list[i],
						   E1432_SRCBUFFER_INIT_EMPTY));
		}	

		/* down load arb source buffer A signal */
		/* (void) printf("preload buffer A\n"); */
		arb_xfrsize2 = arb_xfrsize;
		for (k=0; k<arb_srcbufsize;k = k + arb_xfrsize)
		{	
		    arb_srcbufleft = arb_srcbufsize - k;
		    if (arb_srcbufleft < arb_xfrsize) arb_xfrsize2 = arb_srcbufleft;
		    for (i=0; i<src_count;i++)
		    {	
#ifdef DPRINT
			(void) printf("preload buffer A, chan %x, %d words\n",src_list[i],arb_xfrsize2);
#endif
			xfr_src_sig(hw, src_list[i], src_sigbuffptrs_a[i]+k,arb_xfrsize2,arb_xfrmodewait);
	    }
		}
		if (arb_xfrmode == E1432_SRC_DATA_MODE_AB)
		{
		    /* (void) printf("preload buffer B\n"); */
		    /* down load arb source buffer B signal */
		    arb_xfrsize2 = arb_xfrsize;
		    for (k=0; k<arb_srcbufsize;k = k + arb_xfrsize)
		    {	
			arb_srcbufleft = arb_srcbufsize - k;
			if (arb_srcbufleft < arb_xfrsize) arb_xfrsize2 = arb_srcbufleft;
			for (i=0; i<src_count;i++)
			{	
#ifdef DPRINT
			    (void) printf("preload buffer B, chan %x, %d words\n",src_list[i],arb_xfrsize2);
#endif
	    
			    xfr_src_sig(hw, src_list[i],
					src_sigbuffptrs_b[i]+k,arb_xfrsize2,arb_xfrmodewait);
		    
			}
		    }
		}
#ifdef DPRINT
		(void) printf("done pre-loading buffer(s)\n");
#endif
	    }
	    else signal = 0;
	    
#ifdef DPRINT
	    (void) printf("arm_measure \n");
#endif
	    CHECK(e1432_arm_measure(hw, group, wait));
#ifdef DPRINT
	    (void) printf("trigger_measure \n");
#endif
	    CHECK(e1432_trigger_measure(hw, group, nowait));
	}

/* Wait for group block ready */
	error = 0;
	while (!error) /* Note ; */
	{
	    if (in_count < 1)
	    {
		/* no inputs wait for meas state = idle */
check_meas_state:
		CHECK(e1432_get_meas_state(hw, group, &meas_state));
#ifdef DPRINT
		(void) printf("meas_state %x\n",meas_state);
#endif
		if (meas_state == E1432_MEAS_STATE_IDLE) break;
		goto check_meas_state;
	    }
	    
#ifdef DPRINT
	    (void) printf("checking block_available \n");
#endif
	    error = e1432_block_available(hw, in_group);
	    if (error < 0)
	    {
		(void) printf("e1432_block_available() failed, error: %d\n", error);
		return -1;
	    }
	    if (error > 0)
	    {
		CHECK(read_data());
		break;
	    }
	}	
    }

    /*NOTREACHED*/
	return 0;
}
